<?php
/**
 * WooCommerce Compatibility File
 *
 * @package HolaFone
 */

if (!defined('ABSPATH')) {
    exit;
}

// Declare WooCommerce support
function holafone_woocommerce_setup() {
    add_theme_support('woocommerce', array(
        'thumbnail_image_width' => 400,
        'single_image_width'    => 600,
        'product_grid'          => array(
            'default_rows'    => 3,
            'min_rows'        => 1,
            'default_columns' => 4,
            'min_columns'     => 1,
            'max_columns'     => 6,
        ),
    ));
    add_theme_support('wc-product-gallery-zoom');
    add_theme_support('wc-product-gallery-lightbox');
    add_theme_support('wc-product-gallery-slider');
}
add_action('after_setup_theme', 'holafone_woocommerce_setup');

// Remove default WooCommerce wrapper
remove_action('woocommerce_before_main_content', 'woocommerce_output_content_wrapper', 10);
remove_action('woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end', 10);

// Add custom wrapper
function holafone_woocommerce_wrapper_start() {
    echo '<div class="container my-5"><div class="row"><div class="col-12">';
}
add_action('woocommerce_before_main_content', 'holafone_woocommerce_wrapper_start', 10);

function holafone_woocommerce_wrapper_end() {
    echo '</div></div></div>';
}
add_action('woocommerce_after_main_content', 'holafone_woocommerce_wrapper_end', 10);

// Customize product loop
function holafone_woocommerce_loop_product_title() {
    echo '<h3 class="woocommerce-loop-product__title"><a href="' . get_permalink() . '">' . get_the_title() . '</a></h3>';
}
remove_action('woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_title', 10);
add_action('woocommerce_shop_loop_item_title', 'holafone_woocommerce_loop_product_title', 10);

// Add discount badge to products
function holafone_add_discount_badge() {
    global $product;
    
    if ($product->is_on_sale()) {
        $regular_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();
        
        if ($regular_price && $sale_price) {
            $discount_percent = round((($regular_price - $sale_price) / $regular_price) * 100);
            echo '<div class="discount-badge position-absolute" style="top: 15px; right: 15px; z-index: 10;">';
            echo '<span class="discount-percent">' . $discount_percent . '%</span>';
            echo '<span class="discount-text">LESS</span>';
            echo '</div>';
        }
    }
}
add_action('woocommerce_before_shop_loop_item_title', 'holafone_add_discount_badge', 5);

// Customize cart icon in header
function holafone_cart_link() {
    if (class_exists('WooCommerce')) {
        $cart_count = WC()->cart->get_cart_contents_count();
        $cart_url = wc_get_cart_url();
        
        echo '<a href="' . esc_url($cart_url) . '" class="cart-contents position-relative">';
        echo '<i class="fas fa-shopping-cart"></i>';
        if ($cart_count > 0) {
            echo '<span class="cart-count badge bg-danger position-absolute top-0 start-100 translate-middle">' . $cart_count . '</span>';
        }
        echo '</a>';
    }
}

// Update cart count via AJAX
function holafone_update_cart_count() {
    if (class_exists('WooCommerce')) {
        wp_die(WC()->cart->get_cart_contents_count());
    }
    wp_die('0');
}
add_action('wp_ajax_update_cart_count', 'holafone_update_cart_count');
add_action('wp_ajax_nopriv_update_cart_count', 'holafone_update_cart_count');

// Customize breadcrumbs
function holafone_woocommerce_breadcrumbs() {
    return array(
        'delimiter'   => ' / ',
        'wrap_before' => '<nav class="woocommerce-breadcrumb breadcrumb">',
        'wrap_after'  => '</nav>',
        'before'      => '<span class="breadcrumb-item">',
        'after'       => '</span>',
        'home'        => _x('Home', 'breadcrumb', 'holafone'),
    );
}
add_filter('woocommerce_breadcrumb_defaults', 'holafone_woocommerce_breadcrumbs');

// Customize product tabs
function holafone_woocommerce_product_tabs($tabs) {
    // Rename the description tab
    $tabs['description']['title'] = __('Product Details', 'holafone');
    
    // Add custom tab
    $tabs['refurbishment_info'] = array(
        'title'    => __('Refurbishment Info', 'holafone'),
        'priority' => 50,
        'callback' => 'holafone_refurbishment_info_tab_content'
    );
    
    return $tabs;
}
add_filter('woocommerce_product_tabs', 'holafone_woocommerce_product_tabs', 98);

function holafone_refurbishment_info_tab_content() {
    echo '<h3>Refurbishment Process</h3>';
    echo '<p>All our refurbished devices go through a comprehensive testing and refurbishment process to ensure they meet our high quality standards.</p>';
    echo '<ul>';
    echo '<li>Complete functional testing</li>';
    echo '<li>Professional cleaning and sanitization</li>';
    echo '<li>Battery health verification</li>';
    echo '<li>Quality assurance checks</li>';
    echo '<li>Warranty coverage included</li>';
    echo '</ul>';
}

// Customize checkout fields
function holafone_checkout_fields($fields) {
    // Add Bootstrap classes to checkout fields
    foreach ($fields as $fieldset_key => $fieldset) {
        foreach ($fieldset as $field_key => $field) {
            $fields[$fieldset_key][$field_key]['input_class'] = array('form-control');
            $fields[$fieldset_key][$field_key]['label_class'] = array('form-label');
        }
    }
    
    return $fields;
}
add_filter('woocommerce_checkout_fields', 'holafone_checkout_fields');

// Customize single product layout
function holafone_single_product_summary() {
    remove_action('woocommerce_single_product_summary', 'woocommerce_template_single_rating', 10);
    add_action('woocommerce_single_product_summary', 'woocommerce_template_single_rating', 15);
}
add_action('init', 'holafone_single_product_summary');

// Add custom product fields
function holafone_product_custom_fields() {
    global $post;
    
    // Add custom fields for refurbished products
    woocommerce_wp_select(array(
        'id'      => '_refurbishment_grade',
        'label'   => __('Refurbishment Grade', 'holafone'),
        'options' => array(
            ''          => __('Select grade...', 'holafone'),
            'excellent' => __('Excellent', 'holafone'),
            'very-good' => __('Very Good', 'holafone'),
            'good'      => __('Good', 'holafone'),
            'fair'      => __('Fair', 'holafone'),
        )
    ));
    
    woocommerce_wp_text_input(array(
        'id'          => '_battery_health',
        'label'       => __('Battery Health (%)', 'holafone'),
        'placeholder' => '85',
        'type'        => 'number',
        'custom_attributes' => array(
            'min' => '0',
            'max' => '100'
        )
    ));
    
    woocommerce_wp_checkbox(array(
        'id'    => '_includes_charger',
        'label' => __('Includes Charger', 'holafone'),
    ));
}
add_action('woocommerce_product_options_general_product_data', 'holafone_product_custom_fields');

// Save custom product fields
function holafone_save_product_custom_fields($post_id) {
    $refurbishment_grade = $_POST['_refurbishment_grade'];
    if (!empty($refurbishment_grade)) {
        update_post_meta($post_id, '_refurbishment_grade', esc_attr($refurbishment_grade));
    }
    
    $battery_health = $_POST['_battery_health'];
    if (!empty($battery_health)) {
        update_post_meta($post_id, '_battery_health', esc_attr($battery_health));
    }
    
    $includes_charger = isset($_POST['_includes_charger']) ? 'yes' : 'no';
    update_post_meta($post_id, '_includes_charger', $includes_charger);
}
add_action('woocommerce_process_product_meta', 'holafone_save_product_custom_fields');

// Display custom fields on single product page
function holafone_display_custom_product_fields() {
    global $product;
    
    $refurbishment_grade = get_post_meta($product->get_id(), '_refurbishment_grade', true);
    $battery_health = get_post_meta($product->get_id(), '_battery_health', true);
    $includes_charger = get_post_meta($product->get_id(), '_includes_charger', true);
    
    if ($refurbishment_grade || $battery_health || $includes_charger) {
        echo '<div class="product-custom-info mt-3">';
        
        if ($refurbishment_grade) {
            echo '<p><strong>Grade:</strong> ' . ucfirst(str_replace('-', ' ', $refurbishment_grade)) . '</p>';
        }
        
        if ($battery_health) {
            echo '<p><strong>Battery Health:</strong> ' . $battery_health . '%</p>';
        }
        
        if ($includes_charger === 'yes') {
            echo '<p><strong>Includes:</strong> Charger</p>';
        }
        
        echo '</div>';
    }
}
add_action('woocommerce_single_product_summary', 'holafone_display_custom_product_fields', 25);

// Customize shop page columns
function holafone_shop_columns() {
    return 4; // 4 products per row
}
add_filter('loop_shop_columns', 'holafone_shop_columns');

// Customize products per page
function holafone_products_per_page() {
    return 12;
}
add_filter('loop_shop_per_page', 'holafone_products_per_page', 20);
