<?php
/**
 * HolaFone functions and definitions
 *
 * @package HolaFone
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

// Theme setup
function holafone_setup() {
    // Add default posts and comments RSS feed links to head.
    add_theme_support('automatic-feed-links');

    // Let WordPress manage the document title.
    add_theme_support('title-tag');

    // Enable support for Post Thumbnails on posts and pages.
    add_theme_support('post-thumbnails');

    // Add support for custom logo
    add_theme_support('custom-logo', array(
        'height'      => 50,
        'width'       => 50,
        'flex-width'  => true,
        'flex-height' => true,
    ));

    // This theme uses wp_nav_menu() in multiple locations.
    register_nav_menus(array(
        'primary' => esc_html__('Primary Menu', 'holafone'),
        'top-menu' => esc_html__('Top Menu', 'holafone'),
        'footer-links' => esc_html__('Footer Links', 'holafone'),
        'footer-legal' => esc_html__('Footer Legal', 'holafone'),
    ));

    // Switch default core markup for search form, comment form, and comments
    add_theme_support('html5', array(
        'search-form',
        'comment-form',
        'comment-list',
        'gallery',
        'caption',
    ));

    // Add theme support for selective refresh for widgets.
    add_theme_support('customize-selective-refresh-widgets');

    // WooCommerce support
    add_theme_support('woocommerce');
    add_theme_support('wc-product-gallery-zoom');
    add_theme_support('wc-product-gallery-lightbox');
    add_theme_support('wc-product-gallery-slider');
}
add_action('after_setup_theme', 'holafone_setup');

// Enqueue scripts and styles
function holafone_scripts() {
    // Bootstrap CSS
    wp_enqueue_style('bootstrap', 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css', array(), '5.3.0');
    
    // Font Awesome
    wp_enqueue_style('font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css', array(), '6.0.0');
    
    // Google Fonts
    wp_enqueue_style('google-fonts', 'https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap', array(), null);
    
    // Theme stylesheet
    wp_enqueue_style('holafone-style', get_stylesheet_uri(), array('bootstrap'), wp_get_theme()->get('Version'));

    // Bootstrap JS
    wp_enqueue_script('bootstrap', 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js', array(), '5.3.0', true);
    
    // Theme JS
    wp_enqueue_script('holafone-script', get_template_directory_uri() . '/assets/js/theme.js', array('jquery', 'bootstrap'), wp_get_theme()->get('Version'), true);

    // Localize script for AJAX
    wp_localize_script('holafone-script', 'holafone_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('holafone_nonce')
    ));
}
add_action('wp_enqueue_scripts', 'holafone_scripts');

// Bootstrap Nav Walker
require_once get_template_directory() . '/inc/bootstrap-nav-walker.php';

// Customizer
require_once get_template_directory() . '/inc/customizer.php';

// WooCommerce customizations
if (class_exists('WooCommerce')) {
    require_once get_template_directory() . '/inc/woocommerce.php';
}

// Newsletter signup AJAX handler
function holafone_newsletter_signup() {
    check_ajax_referer('newsletter_signup', 'newsletter_nonce');
    
    $email = sanitize_email($_POST['email']);
    $marketing_consent = isset($_POST['marketing_consent']) ? true : false;
    
    if (!is_email($email)) {
        wp_die('Invalid email address');
    }
    
    // Here you can integrate with your email service provider
    // For now, we'll just store in WordPress options or custom table
    
    $subscribers = get_option('holafone_newsletter_subscribers', array());
    
    if (!in_array($email, $subscribers)) {
        $subscribers[] = array(
            'email' => $email,
            'consent' => $marketing_consent,
            'date' => current_time('mysql')
        );
        update_option('holafone_newsletter_subscribers', $subscribers);
    }
    
    wp_die('Thank you for subscribing!');
}
add_action('wp_ajax_newsletter_signup', 'holafone_newsletter_signup');
add_action('wp_ajax_nopriv_newsletter_signup', 'holafone_newsletter_signup');

// Remove WooCommerce default styles
add_filter('woocommerce_enqueue_styles', '__return_empty_array');

// Custom excerpt length
function holafone_excerpt_length($length) {
    return 20;
}
add_filter('excerpt_length', 'holafone_excerpt_length', 999);

// Custom excerpt more
function holafone_excerpt_more($more) {
    return '...';
}
add_filter('excerpt_more', 'holafone_excerpt_more');

// Add custom image sizes
function holafone_image_sizes() {
    add_image_size('product-thumb', 400, 400, true);
    add_image_size('category-thumb', 378, 338, true);
    add_image_size('hero-banner', 1920, 600, true);
}
add_action('after_setup_theme', 'holafone_image_sizes');

// Widget areas
function holafone_widgets_init() {
    register_sidebar(array(
        'name'          => esc_html__('Sidebar', 'holafone'),
        'id'            => 'sidebar-1',
        'description'   => esc_html__('Add widgets here.', 'holafone'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h2 class="widget-title">',
        'after_title'   => '</h2>',
    ));
}
add_action('widgets_init', 'holafone_widgets_init');

// Custom post types for testimonials, FAQs, etc.
function holafone_custom_post_types() {
    // FAQ Post Type
    register_post_type('faq', array(
        'labels' => array(
            'name' => 'FAQs',
            'singular_name' => 'FAQ',
            'add_new' => 'Add New FAQ',
            'add_new_item' => 'Add New FAQ',
            'edit_item' => 'Edit FAQ',
            'new_item' => 'New FAQ',
            'view_item' => 'View FAQ',
            'search_items' => 'Search FAQs',
            'not_found' => 'No FAQs found',
            'not_found_in_trash' => 'No FAQs found in trash'
        ),
        'public' => false,
        'show_ui' => true,
        'show_in_menu' => true,
        'supports' => array('title', 'editor', 'page-attributes'),
        'menu_icon' => 'dashicons-editor-help'
    ));
}
add_action('init', 'holafone_custom_post_types');

// Security enhancements
function holafone_security_headers() {
    if (!is_admin()) {
        header('X-Content-Type-Options: nosniff');
        header('X-Frame-Options: SAMEORIGIN');
        header('X-XSS-Protection: 1; mode=block');
    }
}
add_action('send_headers', 'holafone_security_headers');

// Optimize loading
function holafone_optimize_loading() {
    // Remove emoji scripts
    remove_action('wp_head', 'print_emoji_detection_script', 7);
    remove_action('wp_print_styles', 'print_emoji_styles');
    
    // Remove WordPress version
    remove_action('wp_head', 'wp_generator');
    
    // Remove RSD link
    remove_action('wp_head', 'rsd_link');
    
    // Remove wlwmanifest link
    remove_action('wp_head', 'wlwmanifest_link');
}
add_action('init', 'holafone_optimize_loading');

// Custom login logo
function holafone_login_logo() {
    $custom_logo_id = get_theme_mod('custom_logo');
    if ($custom_logo_id) {
        $logo = wp_get_attachment_image_src($custom_logo_id, 'full');
        echo '<style type="text/css">
            #login h1 a, .login h1 a {
                background-image: url(' . $logo[0] . ');
                height: 80px;
                width: 320px;
                background-size: contain;
                background-repeat: no-repeat;
                padding-bottom: 30px;
            }
        </style>';
    }
}
add_action('login_enqueue_scripts', 'holafone_login_logo');

// Custom admin footer
function holafone_admin_footer() {
    echo 'Thank you for using <strong>HolaFone Theme</strong>.';
}
add_filter('admin_footer_text', 'holafone_admin_footer');
